/*:
 * @target MZ
 * @plugindesc v1.0 オプション画面に説明ウィンドウを追加。項目ごとに説明文（文字/変数）を表示
 * @author HS
 *
 * @param Position
 * @text 配置
 * @type select
 * @option Bottom（下部に横長）
 * @value bottom
 * @option Right（右側に縦長）
 * @value right
 * @default bottom
 *
 * @param HelpHeight
 * @text 高さ（Bottom時）
 * @type number
 * @min 60
 * @default 120
 *
 * @param HelpWidth
 * @text 幅（Right時）
 * @type number
 * @min 160
 * @default 360
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @min 12
 * @default 20
 *
 * @param LinePad
 * @text 行パディング
 * @type number
 * @min 0
 * @default 6
 *
 * @param ParseEscape
 * @text 制御文字(\C,\I,\V)を解釈
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 *
 * @param VisibleByDefault
 * @text 初期表示
 * @type boolean
 * @on 表示する
 * @off 非表示
 * @default true
 *
 * @param fastModeText
 * @text 説明文：高速モード（テキスト）
 * @type string
 * @default テキストや演出の待機を短縮します。
 *
 * @param fastModeVar
 * @text 説明文：高速モード（変数ID・0で無効）
 * @type variable
 * @default 0
 *
 * @param hsAutoText
 * @text 説明文：自動送り（テキスト）
 * @type string
 * @default メッセージを一定時間で自動で進めます。
 *
 * @param hsAutoVar
 * @text 説明文：自動送り（変数ID・0で無効）
 * @type variable
 * @default 0
 *
 * @param hsProtectText
 * @text 説明文：重要シーンを保護（テキスト）
 * @type string
 * @default ONにすると重要イベント中は高速/自動送りを停止します。OFFにすると重要イベントも流れます（自己責任）。
 *
 * @param hsProtectVar
 * @text 説明文：重要シーンを保護（変数ID・0で無効）
 * @type variable
 * @default 0
 *
 * @command SetHelpText
 * @text 説明文を設定（テキスト）
 * @desc 任意のオプション項目に説明文（テキスト）を登録します
 * @arg symbol
 * @text 項目シンボル
 * @type string
 * @default
 * @arg text
 * @text 説明テキスト
 * @type string
 * @default
 *
 * @command SetHelpVar
 * @text 説明文を設定（変数）
 * @desc 任意のオプション項目に説明文（変数ID）を登録します
 * @arg symbol
 * @text 項目シンボル
 * @type string
 * @default
 * @arg varId
 * @text 変数ID
 * @type variable
 * @default 0
 *
 * @command SetVisible
 * @text 表示切替
 * @desc ヘルプウィンドウの表示/非表示を切り替えます
 * @arg visible
 * @text 表示する
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 */
(() => {
  "use strict";
  const PN  = "HS_OptionHelp";
  const PRM = PluginManager.parameters(PN);

  const POS      = String(PRM["Position"] || "bottom");
  const HELP_H   = Number(PRM["HelpHeight"] || 120);
  const HELP_W   = Number(PRM["HelpWidth"] || 360);
  const FONTSIZE = Number(PRM["FontSize"] || 20);
  const LINEPAD  = Number(PRM["LinePad"] || 6);
  const ESCAPE   = String(PRM["ParseEscape"] || "true") === "true";
  let   VISIBLE  = String(PRM["VisibleByDefault"] || "true") === "true";

  const defaults = {
    fastMode:  { text: String(PRM["fastModeText"]||""),   varId: Number(PRM["fastModeVar"]||0) },
    hsAuto:    { text: String(PRM["hsAutoText"]||""),     varId: Number(PRM["hsAutoVar"]||0) },
    hsProtect: { text: String(PRM["hsProtectText"]||""),  varId: Number(PRM["hsProtectVar"]||0) },
  };

  const map = {}; // symbol -> {text?, varId?}

  // Plugin Commands
  PluginManager.registerCommand(PN, "SetHelpText", args => {
    const sym = String(args.symbol||"");
    const txt = String(args.text||"");
    if (!sym) return;
    map[sym] = map[sym] || {};
    map[sym].text  = txt;
    map[sym].varId = 0;
  });

  PluginManager.registerCommand(PN, "SetHelpVar", args => {
    const sym = String(args.symbol||"");
    const vid = Number(args.varId||0);
    if (!sym) return;
    map[sym] = map[sym] || {};
    map[sym].varId = vid;
  });

  PluginManager.registerCommand(PN, "SetVisible", args => {
    VISIBLE = String(args.visible||"true")==="true";
    const sc = SceneManager._scene;
    if (sc && sc._hsOptionHelpWindow) sc._hsOptionHelpWindow.visible = VISIBLE;
  });

  // ★修正ポイント：タイトル時の 0（数値）を不採用にし、文字列のみ採用
  function textForSymbol(symbol){
    const src = map[symbol] || defaults[symbol] || null;
    if (!src) return "";
    if (src.varId && window.$gameVariables) {
      const v = $gameVariables.value(src.varId);
      if (typeof v === "string" && v.trim() !== "") return String(v);
    }
    return src.text || "";
  }

  // Help Window
  class Window_OptionHelp extends Window_Base {
    initialize(rect) {
      super.initialize(rect);
      this._text = "";
      this.openness = 255;
    }
    resetFontSettings(){
      super.resetFontSettings();
      this.contents.fontSize = FONTSIZE;
    }
    lineHeight(){ return Math.max(16, FONTSIZE + LINEPAD); }
    setText(t){
      const s = String(t||"");
      if (this._text !== s){
        this._text = s;
        this.refresh();
      }
    }
    refresh(){
      this.contents.clear();
      this.resetFontSettings();
      const t = ESCAPE ? this.convertEscapeCharacters(this._text) : this._text;
      // 改行で分割して描画
      const lines = String(t).split(/\r?\n/);
      let y = this.padding;
      for (const ln of lines){
        this.drawTextEx(ln, this.padding, y, this.contentsWidth());
        y += this.lineHeight();
      }
    }
  }

  // Scene_Options 拡張
  const _Scene_Options_create = Scene_Options.prototype.create;
  Scene_Options.prototype.create = function(){
    _Scene_Options_create.call(this);
    this.createHsOptionHelp();
    this.updateHsOptionHelp(); // 初期行の説明
  };

  Scene_Options.prototype.createHsOptionHelp = function(){
    const orect = this.optionsWindowRect();
    let rect;
    if (POS === "right"){
      const x = orect.x + orect.width;
      rect = new Rectangle(x, orect.y, HELP_W, orect.height);
    } else { // bottom
      const x = 0;
      const y = Graphics.boxHeight - HELP_H;
      rect = new Rectangle(x, y, Graphics.boxWidth, HELP_H);
    }
    this._hsOptionHelpWindow = new Window_OptionHelp(rect);
    this._hsOptionHelpWindow.visible = VISIBLE;
    this.addWindow(this._hsOptionHelpWindow);

    // 選択変更で更新
    const win = this._optionsWindow;
    const _select = win.select;
    win.select = function(index){
      _select.call(this, index);
      SceneManager._scene && SceneManager._scene.updateHsOptionHelp();
    };
    // 値変更時も更新（ON/OFFで説明が変わる場合に備え）
    const _change = win.changeValue;
    win.changeValue = function(symbol, value){
      _change.call(this, symbol, value);
      SceneManager._scene && SceneManager._scene.updateHsOptionHelp();
    };
  };

  Scene_Options.prototype.updateHsOptionHelp = function(){
    if (!this._hsOptionHelpWindow) return;
    const sym = this._optionsWindow.commandSymbol(this._optionsWindow.index());
    const txt = textForSymbol(sym);
    this._hsOptionHelpWindow.setText(txt);
  };
})();
